<?php

namespace App\Controllers;

use App\Models\UserModel;
use App\Models\PelangganModel;
use App\Models\PesananLayananModel;
use App\Models\TeknisiModel;
use App\Models\PenugasanModel;
use App\Models\BillingModel;

class Admin extends BaseController
{
    protected $userModel;
    protected $pelangganModel;
    protected $pesananModel;
    protected $teknisiModel;
    protected $penugasanModel;
    protected $billingModel;

    public function __construct()
    {
        $this->userModel = new UserModel();
        $this->pelangganModel = new PelangganModel();
        $this->pesananModel = new PesananLayananModel();
        $this->teknisiModel = new TeknisiModel();
        $this->penugasanModel = new PenugasanModel();
        $this->billingModel = new BillingModel();
    }

    public function dashboard()
    {
        // Check admin authentication
        if (!session()->get('logged_in') || session()->get('role') !== 'admin') {
            return redirect()->to('/auth/login');
        }

        // Get dashboard statistics
        $data = [
            'title' => 'Dashboard Admin',
            'total_teknisi' => $this->teknisiModel->countAll(),
            'total_pelanggan' => $this->pelangganModel->countAll(),
            'pesanan_baru' => $this->pesananModel->where('status_pesanan', 'baru')->countAllResults(),
            'tugas_berjalan' => $this->penugasanModel->where('status_tugas', 'dikerjakan')->countAllResults(),
            'tugas_hari_ini' => $this->penugasanModel->getPenugasanByTanggal(date('Y-m-d'))
        ];

        return view('admin/dashboard', $data);
    }

    public function pelanggan()
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'admin') {
            return redirect()->to('/auth/login');
        }

        $data = [
            'title' => 'Manajemen Pelanggan',
            'pelanggan' => $this->pelangganModel->getPelangganWithLastService()
        ];

        return view('admin/pelanggan', $data);
    }

    public function addPelanggan()
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'admin') {
            return redirect()->to('/auth/login');
        }

        if ($this->request->getMethod() === 'POST') {
            $rules = [
                'nama_pelanggan' => 'required|max_length[100]',
                'no_whatsapp' => 'required|max_length[20]',
                'alamat_lengkap' => 'required'
            ];

            if (!$this->validate($rules)) {
                return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
            }

            $data = [
                'nama_pelanggan' => $this->request->getPost('nama_pelanggan'),
                'no_whatsapp' => $this->request->getPost('no_whatsapp'),
                'alamat_lengkap' => $this->request->getPost('alamat_lengkap'),
                'koordinat_gps' => $this->request->getPost('koordinat_gps')
            ];

            if ($this->pelangganModel->insert($data)) {
                return redirect()->to('/admin/pelanggan')->with('success', 'Pelanggan berhasil ditambahkan.');
            } else {
                return redirect()->back()->with('error', 'Gagal menambahkan pelanggan.');
            }
        }

        $data = ['title' => 'Tambah Pelanggan'];
        return view('admin/add_pelanggan', $data);
    }

    public function pesanan()
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'admin') {
            return redirect()->to('/auth/login');
        }

        $data = [
            'title' => 'Manajemen Pesanan',
            'pesanan' => $this->pesananModel->getPesananWithPelanggan()
        ];

        return view('admin/pesanan', $data);
    }

    public function addPesanan()
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'admin') {
            return redirect()->to('/auth/login');
        }

        if ($this->request->getMethod() === 'POST') {
            $rules = [
                'pelanggan_id' => 'required|integer',
                'keluhan_awal' => 'required',
                'estimasi_biaya_awal' => 'permit_empty|decimal'
            ];

            if (!$this->validate($rules)) {
                return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
            }

            $data = [
                'pelanggan_id' => $this->request->getPost('pelanggan_id'),
                'keluhan_awal' => $this->request->getPost('keluhan_awal'),
                'estimasi_biaya_awal' => $this->request->getPost('estimasi_biaya_awal') ?: 0,
                'status_pesanan' => 'baru'
            ];

            if ($this->pesananModel->insert($data)) {
                return redirect()->to('/admin/pesanan')->with('success', 'Pesanan berhasil ditambahkan.');
            } else {
                return redirect()->back()->with('error', 'Gagal menambahkan pesanan.');
            }
        }

        $data = [
            'title' => 'Tambah Pesanan',
            'pelanggan' => $this->pelangganModel->findAll()
        ];
        return view('admin/add_pesanan', $data);
    }

    public function teknisi()
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'admin') {
            return redirect()->to('/auth/login');
        }

        $data = [
            'title' => 'Manajemen Teknisi',
            'teknisi' => $this->teknisiModel->getTeknisiWithUser()
        ];

        return view('admin/teknisi', $data);
    }

    public function penjadwalan()
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'admin') {
            return redirect()->to('/auth/login');
        }

        $data = [
            'title' => 'Penjadwalan Tugas',
            'penugasan' => $this->penugasanModel->getPenugasanWithDetails(),
            'pesanan_baru' => $this->pesananModel->getPesananByStatus('baru'),
            'teknisi_tersedia' => $this->teknisiModel->getTeknisiTersedia()
        ];

        return view('admin/penjadwalan', $data);
    }

    public function addPenjadwalan()
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'admin') {
            return redirect()->to('/auth/login');
        }

        if ($this->request->getMethod() === 'POST') {
            $rules = [
                'pesanan_id' => 'required|integer',
                'teknisi_id' => 'required|integer',
                'tanggal_penjadwalan' => 'required|valid_date'
            ];

            if (!$this->validate($rules)) {
                return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
            }

            $data = [
                'pesanan_id' => $this->request->getPost('pesanan_id'),
                'teknisi_id' => $this->request->getPost('teknisi_id'),
                'tanggal_penjadwalan' => $this->request->getPost('tanggal_penjadwalan'),
                'catatan_khusus' => $this->request->getPost('catatan_khusus'),
                'status_tugas' => 'akan_datang'
            ];

            if ($this->penugasanModel->insert($data)) {
                // Update status pesanan menjadi dijadwalkan
                $this->pesananModel->updateStatus($data['pesanan_id'], 'dijadwalkan');
                
                return redirect()->to('/admin/penjadwalan')->with('success', 'Penjadwalan berhasil ditambahkan.');
            } else {
                return redirect()->back()->with('error', 'Gagal menambahkan penjadwalan.');
            }
        }
    }

    public function laporan()
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'admin') {
            return redirect()->to('/auth/login');
        }

        $bulan = $this->request->getGet('bulan');
        $tahun = $this->request->getGet('tahun');

        $data = [
            'title' => 'Laporan',
            'billing' => $this->billingModel->getLaporanPendapatan($bulan, $tahun),
            'bulan' => $bulan,
            'tahun' => $tahun
        ];

        return view('admin/laporan', $data);
    }
}

