<?php

namespace App\Controllers;

use App\Models\TeknisiModel;
use App\Models\PenugasanModel;
use App\Models\BillingModel;

class Teknisi extends BaseController
{
    protected $teknisiModel;
    protected $penugasanModel;
    protected $billingModel;

    public function __construct()
    {
        $this->teknisiModel = new TeknisiModel();
        $this->penugasanModel = new PenugasanModel();
        $this->billingModel = new BillingModel();
    }

    public function dashboard()
    {
        // Check teknisi authentication
        if (!session()->get('logged_in') || session()->get('role') !== 'teknisi') {
            return redirect()->to('/auth/login');
        }

        $teknisiId = session()->get('teknisi_id');
        
        $data = [
            'title' => 'Dashboard Teknisi',
            'tugas_hari_ini' => $this->penugasanModel->getPenugasanByTeknisi($teknisiId),
            'tugas_akan_datang' => $this->penugasanModel->where('teknisi_id', $teknisiId)
                                                      ->where('status_tugas', 'akan_datang')
                                                      ->where('tanggal_penjadwalan >=', date('Y-m-d'))
                                                      ->findAll(),
            'tugas_dikerjakan' => $this->penugasanModel->where('teknisi_id', $teknisiId)
                                                       ->where('status_tugas', 'dikerjakan')
                                                       ->findAll()
        ];

        return view('teknisi/dashboard', $data);
    }

    public function detailTugas($id)
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'teknisi') {
            return redirect()->to('/auth/login');
        }

        $teknisiId = session()->get('teknisi_id');
        
        $tugas = $this->penugasanModel->select('penugasan.*, 
                                              pesanan_layanan.keluhan_awal, 
                                              pelanggan.nama_pelanggan, 
                                              pelanggan.alamat_lengkap, 
                                              pelanggan.no_whatsapp,
                                              pelanggan.koordinat_gps')
                                     ->join('pesanan_layanan', 'pesanan_layanan.id = penugasan.pesanan_id')
                                     ->join('pelanggan', 'pelanggan.id = pesanan_layanan.pelanggan_id')
                                     ->where('penugasan.id', $id)
                                     ->where('penugasan.teknisi_id', $teknisiId)
                                     ->first();

        if (!$tugas) {
            return redirect()->to('/teknisi/dashboard')->with('error', 'Tugas tidak ditemukan.');
        }

        $data = [
            'title' => 'Detail Tugas',
            'tugas' => $tugas
        ];

        return view('teknisi/detail_tugas', $data);
    }

    public function checkIn($id)
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'teknisi') {
            return redirect()->to('/auth/login');
        }

        $teknisiId = session()->get('teknisi_id');
        
        // Verifikasi tugas milik teknisi yang login
        $tugas = $this->penugasanModel->where('id', $id)
                                     ->where('teknisi_id', $teknisiId)
                                     ->first();

        if (!$tugas) {
            return redirect()->to('/teknisi/dashboard')->with('error', 'Tugas tidak ditemukan.');
        }

        if ($tugas['status_tugas'] !== 'akan_datang') {
            return redirect()->to('/teknisi/dashboard')->with('error', 'Tugas sudah dimulai atau selesai.');
        }

        if ($this->penugasanModel->checkIn($id)) {
            return redirect()->to('/teknisi/detail_tugas/' . $id)->with('success', 'Check-in berhasil.');
        } else {
            return redirect()->to('/teknisi/detail_tugas/' . $id)->with('error', 'Gagal melakukan check-in.');
        }
    }

    public function checkOut($id)
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'teknisi') {
            return redirect()->to('/auth/login');
        }

        $teknisiId = session()->get('teknisi_id');

        if ($this->request->getMethod() === 'POST') {
            $rules = [
                'catatan_perbaikan' => 'required',
                'biaya_final' => 'required|decimal'
            ];

            if (!$this->validate($rules)) {
                return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
            }

            $catatan = $this->request->getPost('catatan_perbaikan');
            $biaya = $this->request->getPost('biaya_final');

            if ($this->penugasanModel->checkOut($id, $catatan, $biaya)) {
                // Buat billing otomatis
                $billingData = [
                    'penugasan_id' => $id,
                    'total_tagihan' => $biaya,
                    'metode_pembayaran' => 'belum_dipilih',
                    'status_pembayaran' => 'belum_lunas'
                ];
                $this->billingModel->insert($billingData);

                return redirect()->to('/teknisi/dashboard')->with('success', 'Check-out berhasil dan tagihan telah dibuat.');
            } else {
                return redirect()->back()->with('error', 'Gagal melakukan check-out.');
            }
        }

        // Verifikasi tugas milik teknisi yang login
        $tugas = $this->penugasanModel->where('id', $id)
                                     ->where('teknisi_id', $teknisiId)
                                     ->first();

        if (!$tugas || $tugas['status_tugas'] !== 'dikerjakan') {
            return redirect()->to('/teknisi/dashboard')->with('error', 'Tugas tidak valid untuk check-out.');
        }

        $data = [
            'title' => 'Check-out Tugas',
            'tugas' => $tugas
        ];

        return view('teknisi/checkout', $data);
    }

    public function setPending($id)
    {
        if (!session()->get('logged_in') || session()->get('role') !== 'teknisi') {
            return redirect()->to('/auth/login');
        }

        $teknisiId = session()->get('teknisi_id');

        if ($this->request->getMethod() === 'POST') {
            $rules = [
                'alasan_pending' => 'required',
                'jadwal_lanjutan' => 'required|valid_date'
            ];

            if (!$this->validate($rules)) {
                return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
            }

            $data = [
                'status_tugas' => 'pending',
                'alasan_pending' => $this->request->getPost('alasan_pending'),
                'jadwal_lanjutan' => $this->request->getPost('jadwal_lanjutan')
            ];

            if ($this->penugasanModel->update($id, $data)) {
                return redirect()->to('/teknisi/dashboard')->with('success', 'Tugas berhasil di-pending.');
            } else {
                return redirect()->back()->with('error', 'Gagal mengubah status tugas.');
            }
        }

        // Verifikasi tugas milik teknisi yang login
        $tugas = $this->penugasanModel->where('id', $id)
                                     ->where('teknisi_id', $teknisiId)
                                     ->first();

        if (!$tugas) {
            return redirect()->to('/teknisi/dashboard')->with('error', 'Tugas tidak ditemukan.');
        }

        $data = [
            'title' => 'Set Pending Tugas',
            'tugas' => $tugas
        ];

        return view('teknisi/set_pending', $data);
    }
}

