<?php

namespace App\Models;

use CodeIgniter\Model;

class BillingModel extends Model
{
    protected $table = 'billing';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    protected $allowedFields = [
        'penugasan_id',
        'total_tagihan',
        'metode_pembayaran',
        'status_pembayaran',
        'tanggal_kirim_tagihan',
        'tanggal_lunas'
    ];

    // Dates
    protected $useTimestamps = false;

    // Validation
    protected $validationRules = [
        'penugasan_id' => 'required|integer',
        'total_tagihan' => 'required|decimal',
        'metode_pembayaran' => 'permit_empty|in_list[cash,transfer,belum_dipilih]',
        'status_pembayaran' => 'permit_empty|in_list[belum_lunas,lunas]'
    ];

    protected $validationMessages = [];
    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;

    public function getBillingWithDetails()
    {
        return $this->select('billing.*, 
                             penugasan.tanggal_penjadwalan,
                             pesanan_layanan.keluhan_awal,
                             pelanggan.nama_pelanggan,
                             pelanggan.no_whatsapp,
                             users.nama_lengkap as nama_teknisi')
                    ->join('penugasan', 'penugasan.id = billing.penugasan_id')
                    ->join('pesanan_layanan', 'pesanan_layanan.id = penugasan.pesanan_id')
                    ->join('pelanggan', 'pelanggan.id = pesanan_layanan.pelanggan_id')
                    ->join('teknisi', 'teknisi.id = penugasan.teknisi_id')
                    ->join('users', 'users.id = teknisi.user_id')
                    ->orderBy('billing.id', 'DESC')
                    ->findAll();
    }

    public function getBillingByStatus($status)
    {
        return $this->select('billing.*, 
                             penugasan.tanggal_penjadwalan,
                             pesanan_layanan.keluhan_awal,
                             pelanggan.nama_pelanggan,
                             pelanggan.no_whatsapp,
                             users.nama_lengkap as nama_teknisi')
                    ->join('penugasan', 'penugasan.id = billing.penugasan_id')
                    ->join('pesanan_layanan', 'pesanan_layanan.id = penugasan.pesanan_id')
                    ->join('pelanggan', 'pelanggan.id = pesanan_layanan.pelanggan_id')
                    ->join('teknisi', 'teknisi.id = penugasan.teknisi_id')
                    ->join('users', 'users.id = teknisi.user_id')
                    ->where('billing.status_pembayaran', $status)
                    ->orderBy('billing.id', 'DESC')
                    ->findAll();
    }

    public function updatePembayaran($id, $metode, $status = 'lunas')
    {
        $data = [
            'metode_pembayaran' => $metode,
            'status_pembayaran' => $status
        ];
        
        if ($status === 'lunas') {
            $data['tanggal_lunas'] = date('Y-m-d H:i:s');
        }
        
        return $this->update($id, $data);
    }

    public function kirimTagihan($id)
    {
        return $this->update($id, [
            'tanggal_kirim_tagihan' => date('Y-m-d H:i:s')
        ]);
    }

    public function getLaporanPendapatan($bulan = null, $tahun = null)
    {
        $builder = $this->select('billing.*, 
                                 penugasan.tanggal_penjadwalan,
                                 pelanggan.nama_pelanggan,
                                 users.nama_lengkap as nama_teknisi')
                        ->join('penugasan', 'penugasan.id = billing.penugasan_id')
                        ->join('pesanan_layanan', 'pesanan_layanan.id = penugasan.pesanan_id')
                        ->join('pelanggan', 'pelanggan.id = pesanan_layanan.pelanggan_id')
                        ->join('teknisi', 'teknisi.id = penugasan.teknisi_id')
                        ->join('users', 'users.id = teknisi.user_id')
                        ->where('billing.status_pembayaran', 'lunas');

        if ($bulan && $tahun) {
            $builder->where('MONTH(billing.tanggal_lunas)', $bulan)
                   ->where('YEAR(billing.tanggal_lunas)', $tahun);
        }

        return $builder->orderBy('billing.tanggal_lunas', 'DESC')->findAll();
    }
}

