<?php

namespace App\Models;

use CodeIgniter\Model;

class PenugasanModel extends Model
{
    protected $table = 'penugasan';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    protected $allowedFields = [
        'pesanan_id',
        'teknisi_id',
        'tanggal_penjadwalan',
        'waktu_check_in',
        'waktu_checkout',
        'durasi_kerja_menit',
        'catatan_perbaikan',
        'biaya_final',
        'status_tugas',
        'alasan_pending',
        'jadwal_lanjutan',
        'catatan_khusus'
    ];

    // Dates
    protected $useTimestamps = false;

    // Validation
    protected $validationRules = [
        'pesanan_id' => 'required|integer',
        'teknisi_id' => 'required|integer',
        'tanggal_penjadwalan' => 'required|valid_date',
        'status_tugas' => 'permit_empty|in_list[akan_datang,dikerjakan,pending,selesai]'
    ];

    protected $validationMessages = [];
    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;

    public function getPenugasanWithDetails()
    {
        return $this->select('penugasan.*, 
                             pesanan_layanan.keluhan_awal, 
                             pelanggan.nama_pelanggan, 
                             pelanggan.alamat_lengkap, 
                             pelanggan.no_whatsapp,
                             users.nama_lengkap as nama_teknisi')
                    ->join('pesanan_layanan', 'pesanan_layanan.id = penugasan.pesanan_id')
                    ->join('pelanggan', 'pelanggan.id = pesanan_layanan.pelanggan_id')
                    ->join('teknisi', 'teknisi.id = penugasan.teknisi_id')
                    ->join('users', 'users.id = teknisi.user_id')
                    ->orderBy('penugasan.tanggal_penjadwalan', 'DESC')
                    ->findAll();
    }

    public function getPenugasanByTanggal($tanggal)
    {
        return $this->select('penugasan.*, 
                             pesanan_layanan.keluhan_awal, 
                             pelanggan.nama_pelanggan, 
                             pelanggan.alamat_lengkap, 
                             pelanggan.no_whatsapp,
                             users.nama_lengkap as nama_teknisi')
                    ->join('pesanan_layanan', 'pesanan_layanan.id = penugasan.pesanan_id')
                    ->join('pelanggan', 'pelanggan.id = pesanan_layanan.pelanggan_id')
                    ->join('teknisi', 'teknisi.id = penugasan.teknisi_id')
                    ->join('users', 'users.id = teknisi.user_id')
                    ->where('penugasan.tanggal_penjadwalan', $tanggal)
                    ->findAll();
    }

    public function getPenugasanByTeknisi($teknisiId)
    {
        return $this->select('penugasan.*, 
                             pesanan_layanan.keluhan_awal, 
                             pelanggan.nama_pelanggan, 
                             pelanggan.alamat_lengkap, 
                             pelanggan.no_whatsapp')
                    ->join('pesanan_layanan', 'pesanan_layanan.id = penugasan.pesanan_id')
                    ->join('pelanggan', 'pelanggan.id = pesanan_layanan.pelanggan_id')
                    ->where('penugasan.teknisi_id', $teknisiId)
                    ->orderBy('penugasan.tanggal_penjadwalan', 'DESC')
                    ->findAll();
    }

    public function updateStatus($id, $status, $data = [])
    {
        $updateData = array_merge(['status_tugas' => $status], $data);
        return $this->update($id, $updateData);
    }

    public function checkIn($id)
    {
        return $this->update($id, [
            'waktu_check_in' => date('Y-m-d H:i:s'),
            'status_tugas' => 'dikerjakan'
        ]);
    }

    public function checkOut($id, $catatan = '', $biaya = 0)
    {
        $checkIn = $this->find($id)['waktu_check_in'];
        $checkOut = date('Y-m-d H:i:s');
        
        $durasi = 0;
        if ($checkIn) {
            $durasi = (strtotime($checkOut) - strtotime($checkIn)) / 60; // dalam menit
        }

        return $this->update($id, [
            'waktu_checkout' => $checkOut,
            'durasi_kerja_menit' => $durasi,
            'catatan_perbaikan' => $catatan,
            'biaya_final' => $biaya,
            'status_tugas' => 'selesai'
        ]);
    }
}

